<?php
/**
 * Plugin Name: WooCommerce Bağış Modal
 * Plugin URI: https://www.medyaistan.com
 * Description: Belirli kategorilerdeki ürünler için sepete eklerken bağış modal'ı açar
 * Version: 1.0.0
 * Author: <a href="https://www.medyaistan.com">Medyaistan</a>
 * Author URI: https://www.medyaistan.com
 * Text Domain: wc-donation-modal
 * Domain Path: /languages
 */

// Güvenlik kontrolü
if (!defined('ABSPATH')) {
    exit;
}

// Ana sınıf
class WC_Donation_Modal {
    
    private static $instance = null;
    
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    private function __construct() {
        add_action('init', array($this, 'init'));
    }
    
    public function init() {
        // WooCommerce aktif mi kontrol et
        if (!class_exists('WooCommerce')) {
            add_action('admin_notices', array($this, 'woocommerce_missing_notice'));
            return;
        }
        
        // Hook'ları ekle
        add_action('wp_enqueue_scripts', array($this, 'enqueue_scripts'));
        add_action('wp_ajax_check_donation_category', array($this, 'check_donation_category'));
        add_action('wp_ajax_nopriv_check_donation_category', array($this, 'check_donation_category'));
        add_action('wp_ajax_get_donation_products', array($this, 'get_donation_products'));
        add_action('wp_ajax_nopriv_get_donation_products', array($this, 'get_donation_products'));
        add_action('wp_ajax_add_donation_to_cart', array($this, 'add_donation_to_cart'));
        add_action('wp_ajax_nopriv_add_donation_to_cart', array($this, 'add_donation_to_cart'));
        
        // WooCommerce'in kendi sepete ekleme işlemini kontrol et (sadece çift eklemeyi önlemek için)
        // add_action('woocommerce_add_to_cart_validation', array($this, 'validate_add_to_cart'), 10, 3);
        add_action('woocommerce_checkout_create_order_line_item', array($this, 'save_donation_meta'), 10, 4);
        add_action('woocommerce_admin_order_data_after_billing_address', array($this, 'display_donation_info'));
        add_filter('woocommerce_get_item_data', array($this, 'display_cart_item_donation_data'), 10, 2);
        add_action('woocommerce_order_item_meta_end', array($this, 'display_order_item_donation_data'), 10, 3);
        
        // Admin ayarları
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_init', array($this, 'register_settings'));
    }
    
    public function woocommerce_missing_notice() {
        echo '<div class="notice notice-error"><p><strong>WooCommerce Bağış Modal:</strong> Bu eklenti WooCommerce\'in aktif olmasını gerektirir.</p></div>';
    }
    
    public function enqueue_scripts() {
        if (is_product() || is_shop() || is_product_category()) {
            wp_enqueue_script('wc-donation-modal', plugin_dir_url(__FILE__) . 'assets/js/donation-modal.js', array('jquery'), '1.0.0', true);
            wp_enqueue_style('wc-donation-modal', plugin_dir_url(__FILE__) . 'assets/css/donation-modal.css', array(), '1.0.0');
            
            wp_localize_script('wc-donation-modal', 'wc_donation_ajax', array(
                'ajax_url' => admin_url('admin-ajax.php'),
                'nonce' => wp_create_nonce('wc_donation_nonce'),
                'texts' => array(
                    'modal_title' => 'Bağış Bilgileri',
                    'donation_for_label' => 'Bağış kimin adına yapılacak?',
                    'donation_for_placeholder' => 'Örn: Ahmet Yılmaz',
                    'proxy_label' => 'Vekaletimi veriyorum',
                    'add_to_cart' => 'Sepete Ekle',
                    'cancel' => 'İptal',
                    'required_field' => 'Bu alan zorunludur'
                )
            ));
        }
    }
    
    public function check_donation_category() {
        check_ajax_referer('wc_donation_nonce', 'nonce');
        
        $product_id = intval($_POST['product_id']);
        $donation_categories = get_option('wc_donation_categories', array());
        
        if (empty($donation_categories)) {
            wp_send_json_error('No donation categories set');
        }
        
        $product_categories = wp_get_post_terms($product_id, 'product_cat', array('fields' => 'ids'));
        $is_donation_product = !empty(array_intersect($product_categories, $donation_categories));
        
        wp_send_json_success(array('is_donation_product' => $is_donation_product));
    }
    
    public function get_donation_products() {
        check_ajax_referer('wc_donation_nonce', 'nonce');
        
        $product_ids = isset($_POST['product_ids']) ? array_map('intval', $_POST['product_ids']) : array();
        $donation_categories = get_option('wc_donation_categories', array());
        
        $donation_products = array();
        $normal_products = array();
        
        if (!empty($donation_categories) && !empty($product_ids)) {
            foreach ($product_ids as $product_id) {
                $product_categories = wp_get_post_terms($product_id, 'product_cat', array('fields' => 'ids'));
                $is_donation_product = !empty(array_intersect($product_categories, $donation_categories));
                
                if ($is_donation_product) {
                    $donation_products[] = $product_id;
                } else {
                    $normal_products[] = $product_id;
                }
            }
        } else {
            // Bağış kategorisi yoksa tüm ürünler normal
            $normal_products = $product_ids;
        }
        
        wp_send_json_success(array(
            'donation_products' => $donation_products,
            'normal_products' => $normal_products
        ));
    }
    
    public function validate_add_to_cart($passed, $product_id, $quantity) {
        // Bağış kategorilerini al
        $donation_categories = get_option('wc_donation_categories', array());
        
        if (empty($donation_categories)) {
            return $passed; // Bağış kategorisi yoksa normal işlem
        }
        
        // Ürünün kategorilerini kontrol et
        $product_categories = wp_get_post_terms($product_id, 'product_cat', array('fields' => 'ids'));
        $is_donation_product = !empty(array_intersect($product_categories, $donation_categories));
        
        if ($is_donation_product) {
            // Eğer bizim AJAX'ımızdan geliyorsa izin ver
            if (doing_action('wp_ajax_add_donation_to_cart') || doing_action('wp_ajax_nopriv_add_donation_to_cart')) {
                return $passed;
            }
            
            // Eğer cart item data'da bağış bilgileri varsa izin ver (bizim eklentimizden geliyor)
            if (isset($_POST['donation_for']) || (isset($_REQUEST['cart_item_data']) && is_array($_REQUEST['cart_item_data']))) {
                return $passed;
            }
            
            // Normal sepete ekleme işlemini engelle (sadece frontend'de)
            if (!is_admin()) {
                wc_add_notice('Bu ürün için bağış bilgileri gereklidir. Lütfen modal üzerinden sepete ekleyin.', 'error');
                return false;
            }
        }
        
        return $passed;
    }
    
    public function add_donation_to_cart() {
        try {
            // Nonce kontrolü
            if (!wp_verify_nonce($_POST['nonce'], 'wc_donation_nonce')) {
                wp_send_json(array('success' => false, 'data' => 'Güvenlik kontrolü başarısız'));
                return;
            }
            
            // POST verilerini al
            $product_id = intval($_POST['product_id']);
            $quantity = intval($_POST['quantity']);
            $variation_id = intval($_POST['variation_id']);
            $donation_for = sanitize_text_field($_POST['donation_for']);
            $proxy_given = isset($_POST['proxy_given']) ? 'yes' : 'no';
            
            // Debug bilgileri
            error_log('Donation Modal - Product ID: ' . $product_id);
            error_log('Donation Modal - Donation For: ' . $donation_for);
            
            // Validasyon
            if (empty($donation_for)) {
                wp_send_json(array('success' => false, 'data' => 'Bağış yapılacak kişi adı zorunludur'));
                return;
            }
            
            if ($quantity <= 0) {
                $quantity = 1;
            }
            
            // WooCommerce'in yüklü olduğunu kontrol et
            if (!function_exists('WC') || !WC()->cart) {
                wp_send_json(array('success' => false, 'data' => 'WooCommerce sepet sistemi kullanılamıyor'));
                return;
            }
            
            // Ürünün var olup olmadığını kontrol et
            $product = wc_get_product($product_id);
            if (!$product) {
                wp_send_json(array('success' => false, 'data' => 'Geçersiz ürün ID: ' . $product_id));
                return;
            }
            
            // Bağış kategorisi kontrolü
            $donation_categories = get_option('wc_donation_categories', array());
            if (empty($donation_categories)) {
                wp_send_json(array('success' => false, 'data' => 'Bağış kategorileri ayarlanmamış'));
                return;
            }
            
            $product_categories = wp_get_post_terms($product_id, 'product_cat', array('fields' => 'ids'));
            $is_donation_product = !empty(array_intersect($product_categories, $donation_categories));
            
            if (!$is_donation_product) {
                wp_send_json(array('success' => false, 'data' => 'Bu ürün bağış kategorisinde değil'));
                return;
            }
            
            // Bağış bilgilerini cart item data'ya ekle
            $cart_item_data = array(
                'donation_for' => $donation_for,
                'proxy_given' => $proxy_given
            );
            
            // Sepete ekle
            $cart_item_key = WC()->cart->add_to_cart($product_id, $quantity, $variation_id, array(), $cart_item_data);
            
            if ($cart_item_key) {
                // WooCommerce'in kendi mesaj sistemini kullan
                wc_add_to_cart_message(array($product_id => $quantity), true);
                
                // Başarılı response gönder
                wp_send_json(array(
                    'success' => true,
                    'data' => array(
                        'message' => 'Ürün başarıyla sepete eklendi',
                        'cart_item_key' => $cart_item_key,
                        'donation_for' => $donation_for
                    )
                ));
            } else {
                wp_send_json(array(
                    'success' => false,
                    'data' => 'Ürün sepete eklenemedi - WooCommerce hatası'
                ));
            }
            
        } catch (Exception $e) {
            error_log('Donation Modal Error: ' . $e->getMessage());
            wp_send_json(array('success' => false, 'data' => 'Sistem hatası: ' . $e->getMessage()));
        }
    }
    
    public function save_donation_meta($item, $cart_item_key, $values, $order) {
        if (isset($values['donation_for']) && !empty($values['donation_for'])) {
            $item->add_meta_data('Bağış Adına', $values['donation_for']);
        }
        
        if (isset($values['proxy_given']) && $values['proxy_given'] === 'yes') {
            $item->add_meta_data('Vekalet Verildi', 'Evet');
        }
    }
    
    public function display_donation_info($order) {
        foreach ($order->get_items() as $item) {
            $donation_for = $item->get_meta('Bağış Adına');
            $proxy_given = $item->get_meta('Vekalet Verildi');
            
            if ($donation_for || $proxy_given) {
                echo '<div class="donation-info" style="margin-top: 20px; padding: 10px; background: #f9f9f9; border-left: 4px solid #0073aa;">';
                echo '<h4>Bağış Bilgileri - ' . $item->get_name() . '</h4>';
                
                if ($donation_for) {
                    echo '<p><strong>Bağış Adına:</strong> ' . esc_html($donation_for) . '</p>';
                }
                
                if ($proxy_given) {
                    echo '<p><strong>Vekalet Durumu:</strong> ' . esc_html($proxy_given) . '</p>';
                }
                
                echo '</div>';
            }
        }
    }
    
    public function display_cart_item_donation_data($item_data, $cart_item) {
        if (isset($cart_item['donation_for']) && !empty($cart_item['donation_for'])) {
            $item_data[] = array(
                'key'   => 'Bağış Adına',
                'value' => esc_html($cart_item['donation_for']),
            );
        }
        
        if (isset($cart_item['proxy_given']) && $cart_item['proxy_given'] === 'yes') {
            $item_data[] = array(
                'key'   => 'Vekalet Durumu',
                'value' => 'Vekaletimi veriyorum',
            );
        }
        
        return $item_data;
    }
    
    public function display_order_item_donation_data($item_id, $item, $order) {
        $donation_for = $item->get_meta('Bağış Adına');
        $proxy_given = $item->get_meta('Vekalet Verildi');
        
        if ($donation_for) {
            echo '<div class="wc-order-item-donation-info">';
            echo '<strong>Bağış Adına:</strong> ' . esc_html($donation_for) . '<br>';
            if ($proxy_given) {
                echo '<strong>Vekalet Durumu:</strong> ' . esc_html($proxy_given);
            }
            echo '</div>';
        }
    }
    
    public function add_admin_menu() {
        add_options_page(
            'Bağış Modal Ayarları',
            'Bağış Modal',
            'manage_options',
            'wc-donation-modal',
            array($this, 'admin_page')
        );
    }
    
    public function register_settings() {
        register_setting('wc_donation_modal_settings', 'wc_donation_categories');
    }
    
    public function admin_page() {
        $categories = get_terms(array(
            'taxonomy' => 'product_cat',
            'hide_empty' => false,
        ));
        
        $selected_categories = get_option('wc_donation_categories', array());
        ?>
        <div class="wrap">
            <h1>Bağış Modal Ayarları</h1>
            <form method="post" action="options.php">
                <?php settings_fields('wc_donation_modal_settings'); ?>
                <table class="form-table">
                    <tr>
                        <th scope="row">Bağış Kategorileri</th>
                        <td>
                            <fieldset>
                                <legend class="screen-reader-text"><span>Bağış Kategorileri</span></legend>
                                <?php foreach ($categories as $category): ?>
                                    <label>
                                        <input type="checkbox" name="wc_donation_categories[]" value="<?php echo $category->term_id; ?>" 
                                               <?php checked(in_array($category->term_id, $selected_categories)); ?>>
                                        <?php echo esc_html($category->name); ?>
                                    </label><br>
                                <?php endforeach; ?>
                                <p class="description">Bağış modal'ının açılacağı kategorileri seçin.</p>
                            </fieldset>
                        </td>
                    </tr>
                </table>
                <?php submit_button(); ?>
            </form>
        </div>
        <?php
    }
}

// Eklentiyi başlat
WC_Donation_Modal::get_instance();
