<?php
/**
 * Erken Koruma Sistemi
 * WordPress yüklenmeden önce çalışan dosyaları engeller
 */

if (!defined('ABSPATH')) {
    // Bu dosya WordPress yüklenmeden önce çalışabilir
    // Bu durumda sadece temel kontroller yapılır
    return;
}

class KORUMA_Early_Protection {
    
    private static $instance = null;
    private $suspicious_files = array('hello.php');
    
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    private function __construct() {
        $this->init();
    }
    
    private function init() {
        // Mu-plugins yüklendiğinde kontrol et
        add_action('muplugins_loaded', array($this, 'check_mu_plugins'));
        
        // Plugin yüklendiğinde kontrol et
        add_action('plugins_loaded', array($this, 'check_plugins'), 1);
    }
    
    /**
     * Mu-plugins dosyalarını kontrol et
     */
    public function check_mu_plugins() {
        $mu_plugins_dir = WPMU_PLUGIN_DIR;
        
        if (!is_dir($mu_plugins_dir)) {
            return;
        }
        
        $files = glob($mu_plugins_dir . '/*.php');
        
        foreach ($files as $file) {
            $this->check_file($file);
        }
    }
    
    /**
     * Plugin dosyalarını kontrol et
     */
    public function check_plugins() {
        $plugins_dir = WP_PLUGIN_DIR;
        
        foreach ($this->suspicious_files as $suspicious_file) {
            $file_path = $plugins_dir . '/' . $suspicious_file;
            
            if (file_exists($file_path)) {
                $this->check_file($file_path);
            }
        }
    }
    
    /**
     * Dosyayı kontrol et
     */
    private function check_file($file_path) {
        if (!file_exists($file_path) || !is_readable($file_path)) {
            return;
        }
        
        $content = file_get_contents($file_path);
        $filename = basename($file_path);
        
        // WordPress fonksiyonlarını kullanıyor ama ABSPATH kontrolü yok
        $uses_wp_functions = preg_match('/add_action|add_filter|wp_|get_option|get_plugin_data/', $content);
        $has_abspath_check = preg_match('/if\s*\(\s*!\s*defined\s*\(\s*[\'"]ABSPATH/i', $content);
        
        if ($uses_wp_functions && !$has_abspath_check) {
            $this->log_threat($file_path, 'WordPress fonksiyonları kullanıyor ama ABSPATH kontrolü yok');
        }
        
        // Şüpheli pattern'ler
        $suspicious_patterns = array(
            '/eval\s*\(/i',
            '/base64_decode/i',
            '/stepmomhub\.com/i',
            '/awscloud\.icu/i',
        );
        
        foreach ($suspicious_patterns as $pattern) {
            if (preg_match($pattern, $content)) {
                $this->log_threat($file_path, 'Şüpheli kod pattern\'i tespit edildi: ' . $pattern);
            }
        }
    }
    
    /**
     * Tehdit logla
     */
    private function log_threat($file_path, $reason) {
        // Logger henüz yüklenmemiş olabilir, bu yüzden direkt veritabanına yaz
        global $wpdb;
        
        if (!$wpdb) {
            return;
        }
        
        $table_name = $wpdb->prefix . 'koruma_logs';
        
        // Tablo yoksa oluştur
        if ($wpdb->get_var("SHOW TABLES LIKE '$table_name'") != $table_name) {
            return;
        }
        
        $wpdb->insert(
            $table_name,
            array(
                'event_type' => 'early_threat_detected',
                'message' => sprintf('Erken tehdit tespit edildi: %s - %s', basename($file_path), $reason),
                'data' => maybe_serialize(array('file' => $file_path, 'reason' => $reason)),
                'user_id' => 0,
                'ip_address' => $this->get_user_ip(),
                'created_at' => current_time('mysql')
            )
        );
    }
    
    private function get_user_ip() {
        $ip_keys = array('HTTP_CLIENT_IP', 'HTTP_X_FORWARDED_FOR', 'REMOTE_ADDR');
        foreach ($ip_keys as $key) {
            if (array_key_exists($key, $_SERVER) === true) {
                foreach (explode(',', $_SERVER[$key]) as $ip) {
                    $ip = trim($ip);
                    if (filter_var($ip, FILTER_VALIDATE_IP) !== false) {
                        return $ip;
                    }
                }
            }
        }
        return '0.0.0.0';
    }
}

// WordPress yüklendiyse başlat
if (function_exists('add_action')) {
    KORUMA_Early_Protection::get_instance();
}

